import { useState, useEffect, useRef } from "react";
import { Send, MoreVertical, Search, Menu, X } from "lucide-react";
import { io } from "socket.io-client";

const socket = io("http://localhost:5000");

export default function App() {
  const [message, setMessage] = useState("");
  const [chat, setChat] = useState([]);
  const [isOnline, setIsOnline] = useState(true);
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [username, setUsername] = useState("");
  const [isUsernameSet, setIsUsernameSet] = useState(false);
  const [showMenu, setShowMenu] = useState(false);
  const [showUsersModal, setShowUsersModal] = useState(false);
  const [usersList, setUsersList] = useState([]);
  const chatEndRef = useRef(null);
  const menuRef = useRef(null);

  useEffect(() => {
    socket.on("receiveMessage", (data) => {
      setChat((prev) => [...prev, { ...data, timestamp: new Date() }]);
    });

    socket.on("connect", () => setIsOnline(true));
    socket.on("disconnect", () => setIsOnline(false));

    socket.on("usersList", (users) => {
      setUsersList(users)
    });

    return () => {
      socket.off("receiveMessage");
      socket.off("connect");
      socket.off("disconnect");
      socket.off("usersList");
    };
  }, []);

  useEffect(() => {
    const handleClickOutside = (event) => {
      if (menuRef.current && !menuRef.current.contains(event.target)) {
        setShowMenu(false);
      }
    };

    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  useEffect(() => {
    chatEndRef.current?.scrollIntoView({ behavior: "smooth" });
  }, [chat]);

  const sendMessage = () => {
    if (message.trim()) {
      const newMsg = { text: message, sender: username, timestamp: new Date() };
      socket.emit("sendMessage", newMsg);
      setMessage("");
    }
  };

  const handleKeyPress = (e) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      sendMessage();
    }
  };

  const formatTime = (date) => {
    return new Date(date).toLocaleTimeString("en-US", {
      hour: "2-digit",
      minute: "2-digit",
    });
  };

  const handleLogout = () => {
    setIsUsernameSet(false);
    setUsername("");
    setChat([]);
    setShowMenu(false);
    socket.disconnect();
  };

  const handleViewUsers = () => {
    setShowUsersModal(true);
    setShowMenu(false);
    socket.emit("getUsersList");
  };

  return (
    <>
      {!isUsernameSet ? (
        <div className="flex items-center justify-center min-h-screen bg-gray-900">
          <div className="bg-gray-800 p-8 rounded-2xl shadow-2xl w-full max-w-md">
            <div className="text-center mb-6">
              <div className="w-20 h-20 rounded-full bg-gradient-to-br from-blue-500 to-purple-500 flex items-center justify-center mx-auto mb-4">
                <span className="text-4xl">👤</span>
              </div>
              <h2 className="text-2xl font-bold text-white mb-2">Welcome to Chat</h2>
              <p className="text-gray-400">Enter your name to get started</p>
            </div>
            <input
              type="text"
              value={username}
              onChange={(e) => setUsername(e.target.value)}
              onKeyPress={(e) => {
                if (e.key === "Enter" && username.trim()) {
                  setIsUsernameSet(true);
                  socket.emit("setUsername", username);
                }

              }}
              placeholder="Your name..."
              className="w-full bg-gray-700 text-white rounded-lg px-4 py-3 mb-4 focus:outline-none focus:ring-2 focus:ring-blue-500"
            />
            <button
              // onClick={() => username.trim() && setIsUsernameSet(true)}
              onClick={() => {
                if (username.trim()) {
                  setIsUsernameSet(true);
                  socket.emit("setUsername", username); // SEND TO SERVER
                }
              }}
              disabled={!username.trim()}
              className="w-full bg-blue-600 text-white py-3 rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed font-medium"
            >
              Start Chatting
            </button>
          </div>
        </div>
      ) : (
        <div className="flex h-screen bg-gray-900">
          {/* Sidebar - Mobile Toggle */}
          <div
            className={`fixed inset-0 bg-black bg-opacity-50 z-20 lg:hidden transition-opacity ${sidebarOpen ? "opacity-100" : "opacity-0 pointer-events-none"
              }`}
            onClick={() => setSidebarOpen(false)}
          />

          {/* Sidebar */}
          <div
            className={`fixed lg:static inset-y-0 left-0 w-80 bg-gray-800 border-r border-gray-700 z-30 transform transition-transform lg:translate-x-0 ${sidebarOpen ? "translate-x-0" : "-translate-x-full"
              }`}
          >
            <div className="flex flex-col h-full">
              {/* Sidebar Header */}
              <div className="p-4 border-b border-gray-700">
                <div className="flex items-center justify-between mb-4">
                  <h1 className="text-xl font-bold text-white">Messages</h1>
                  <button
                    onClick={() => setSidebarOpen(false)}
                    className="lg:hidden text-gray-400 hover:text-white"
                  >
                    <X size={24} />
                  </button>
                </div>
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" size={18} />
                  <input
                    type="text"
                    placeholder="Search conversations..."
                    className="w-full bg-gray-700 text-white rounded-lg pl-10 pr-4 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                </div>
              </div>

              {/* Chat List */}
              <div className="flex-1 overflow-y-auto">
                <div className="p-2">
                  <div className="p-3 rounded-lg hover:bg-gray-700 cursor-pointer bg-gray-700">
                    <div className="flex items-center gap-3">
                      <div className="relative">
                        <div className="w-12 h-12 rounded-full bg-gradient-to-br from-blue-500 to-purple-500 flex items-center justify-center text-white font-semibold">
                          GC
                        </div>
                        <div className={`absolute bottom-0 right-0 w-3 h-3 rounded-full border-2 border-gray-800 ${isOnline ? 'bg-green-500' : 'bg-gray-500'}`} />
                      </div>
                      <div className="flex-1 min-w-0">
                        <div className="flex items-center justify-between mb-1">
                          <h3 className="text-white font-medium text-sm">General Chat</h3>
                          <span className="text-xs text-gray-400">Just now</span>
                        </div>
                        <p className="text-xs text-gray-400 truncate">
                          {chat.length > 0 ? chat[chat.length - 1].text : "No messages yet"}
                        </p>
                      </div>
                    </div>
                  </div>
                </div>
              </div>

              {/* User Profile */}
              <div className="p-4 border-t border-gray-700">
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 rounded-full bg-gradient-to-br from-green-500 to-teal-500 flex items-center justify-center text-white font-semibold">
                    {username.slice(0, 2).toUpperCase()}
                  </div>
                  <div className="flex-1">
                    <h3 className="text-white font-medium text-sm">{username}</h3>
                    <p className="text-xs text-gray-400">Online</p>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Main Chat Area */}
          <div className="flex-1 flex flex-col bg-gray-900">
            {/* Chat Header */}
            <div className="bg-gray-800 border-b border-gray-700 px-6 py-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <button
                    onClick={() => setSidebarOpen(true)}
                    className="lg:hidden text-gray-400 hover:text-white mr-2"
                  >
                    <Menu size={24} />
                  </button>
                  <div className="relative">
                    <div className="w-10 h-10 rounded-full bg-gradient-to-br from-blue-500 to-purple-500 flex items-center justify-center text-white font-semibold">
                      GC
                    </div>
                    <div className={`absolute bottom-0 right-0 w-3 h-3 rounded-full border-2 border-gray-800 ${isOnline ? 'bg-green-500' : 'bg-gray-500'}`} />
                  </div>
                  <div>
                    <h2 className="text-white font-semibold">General Chat</h2>
                    <p className="text-xs text-gray-400">
                      {isOnline ? "Active now" : "Disconnected"}
                    </p>
                  </div>
                </div>
                <button className="text-gray-400 hover:text-white relative" ref={menuRef}>
                  <MoreVertical
                    size={20}
                    onClick={() => setShowMenu(!showMenu)}
                    className="cursor-pointer"
                  />

                  {/* Dropdown Menu */}
                  {showMenu && (
                    <div className="absolute right-0 mt-2 w-48 bg-gray-700 rounded-lg shadow-xl border border-gray-600 overflow-hidden z-10">
                      <button
                        onClick={handleViewUsers}
                        className="w-full text-left px-4 py-3 text-white hover:bg-gray-600 transition-colors flex items-center gap-3"
                      >
                        <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z" />
                        </svg>
                        View Users
                      </button>
                      <button
                        onClick={handleLogout}
                        className="w-full text-left px-4 py-3 text-red-400 hover:bg-gray-600 transition-colors flex items-center gap-3"
                      >
                        <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1" />
                        </svg>
                        Logout
                      </button>
                    </div>
                  )}
                </button>
              </div>
            </div>

            {/* Users Modal */}
            {showUsersModal && (
              <div className="fixed inset-0 bg-black/60 flex items-center justify-center z-50 p-4">
                <div className="bg-gray-800 rounded-2xl w-full max-w-md shadow-2xl">
                  <div className="p-6 border-b border-gray-700">
                    <div className="flex items-center justify-between">
                      <h3 className="text-xl font-bold text-white">Active Users</h3>
                      <button
                        onClick={() => setShowUsersModal(false)}
                        className="text-gray-400 hover:text-white"
                      >
                        <X size={24} />
                      </button>
                    </div>
                  </div>
                  <div className="p-4 max-h-96 overflow-y-auto">
                    {usersList.length === 0 ? (
                      <div className="text-center py-8">
                        <p className="text-gray-400">No active users</p>
                      </div>
                    ) : (
                      <div className="space-y-2">
                        {usersList.map((user, index) => (
                          <div
                            key={index}
                            className="flex items-center gap-3 p-3 rounded-lg hover:bg-gray-700 transition-colors"
                          >
                            <div className="w-10 h-10 rounded-full bg-gradient-to-br from-blue-500 to-purple-500 flex items-center justify-center text-white font-semibold">
                              {user.slice(0, 2).toUpperCase()}
                            </div>
                            <div className="flex-1">
                              <h4 className="text-white font-medium">{user}</h4>
                              <div className="flex items-center gap-1.5">
                                <div className="w-2 h-2 rounded-full bg-green-500" />
                                <span className="text-xs text-gray-400">Online</span>
                              </div>
                            </div>
                            {user === username && (
                              <span className="text-xs bg-blue-600 text-white px-2 py-1 rounded-full">
                                You
                              </span>
                            )}
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                </div>
              </div>
            )}

            {/* Messages Area */}
            <div className="flex-1 overflow-y-auto p-6 space-y-4">
              {chat.length === 0 ? (
                <div className="flex items-center justify-center h-full">
                  <div className="text-center">
                    <div className="w-20 h-20 rounded-full bg-gray-800 flex items-center justify-center mx-auto mb-4">
                      <span className="text-4xl">💬</span>
                    </div>
                    <h3 className="text-xl font-semibold text-white mb-2">No messages yet</h3>
                    <p className="text-gray-400">Send a message to start the conversation</p>
                  </div>
                </div>
              ) : (
                chat.map((msg, i) => {
                  const isUser = msg.sender === username;

                  if (msg.type === "system") {
                    return (
                      <div key={i} className="text-center text-sm text-gray-400 italic">
                        {msg.text}
                      </div>
                    );
                  }
                  return (
                    <div
                      key={i}
                      className={`flex items-end gap-2 ${isUser ? "justify-end" : "justify-start"}`}
                    >
                      {!isUser && (
                        <div className="w-8 h-8 rounded-full bg-gradient-to-br from-orange-500 to-pink-500 flex items-center justify-center text-white text-xs font-semibold flex-shrink-0">
                          {msg.sender[0]}
                        </div>
                      )}
                      <div
                        className={`max-w-[70%] rounded-2xl px-4 py-2.5 ${isUser
                          ? "bg-blue-600 text-white"
                          : "bg-gray-800 text-gray-100"
                          }`}
                      >
                        {!isUser && (
                          <p className="text-xs font-semibold mb-1 text-blue-400">
                            {msg.sender}
                          </p>
                        )}
                        <p className="text-sm leading-relaxed break-words">{msg.text}</p>
                        <p className={`text-[10px] mt-1 ${isUser ? "text-blue-200" : "text-gray-500"}`}>
                          {formatTime(msg.timestamp)}
                        </p>
                      </div>
                      {isUser && (
                        <div className="w-8 h-8 rounded-full bg-gradient-to-br from-green-500 to-teal-500 flex items-center justify-center text-white text-xs font-semibold flex-shrink-0">
                          {username.slice(0, 2).toUpperCase()}
                        </div>
                      )}
                    </div>
                  );
                })
              )}
              <div ref={chatEndRef} />
            </div>

            {/* Input Area */}
            <div className="bg-gray-800 border-t border-gray-700 px-6 py-4">
              <div className="flex gap-3 items-end">
                <textarea
                  className="flex-1 bg-gray-700 text-white rounded-2xl px-4 py-3 resize-none focus:outline-none focus:ring-2 focus:ring-blue-500 max-h-32"
                  value={message}
                  onChange={(e) => setMessage(e.target.value)}
                  onKeyPress={handleKeyPress}
                  placeholder="Type a message..."
                  rows={1}
                />
                <button
                  onClick={sendMessage}
                  disabled={!message.trim()}
                  className="bg-blue-600 text-white p-3 rounded-full hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex-shrink-0"
                >
                  <Send size={20} />
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </>
  );
}